#include <iostream>
#include <cmath>
#include <chrono>
#include <vector>
#include "TRandom.h"
#include "TRandom1.h"
#include "TRandom2.h"
#include "TRandom3.h"
#include "TMath.h"
#include "TGraph.h"
#include "TH1F.h"
#include "TCanvas.h"
#include "TApplication.h"

unsigned long long int seeddefault = 1;

class RndGen;
void testperiod(RndGen* rndgen, int algo, unsigned long long int seed=seeddefault);
void testmarsaglia(RndGen* rndgen, int algo, unsigned long long int seed=seeddefault);
void testarbitrary(RndGen* rndgen, int algo, int ifunc, double par0, double par1, unsigned long long int entries, unsigned long long int seed=seeddefault);

class RndGen {
  
private:
  unsigned long long int random;

  //********** only valid for LCG ****************************
  // buon generatore
  // const unsigned long long int m=pow(2,32);
  // const unsigned long long int lambda=9;
  // const unsigned long long int mu=3;

  // genera la sequenza 0,1,0,1,0,....
  // const unsigned long long int      m=2;
  // const unsigned long long int lambda=1;
  // const unsigned long long int     mu=1;

  // genera la sequenza 0,1,0,1,0,.... (non rispetta lambda<m e mu<m)
  // const unsigned long long int      m=2;
  // const unsigned long long int lambda=9;
  // const unsigned long long int     mu=3;
  
  // generatore "ragionevole" ma che presenta l'effetto Marsaglia
  // const unsigned long long int m=pow(2,12);
  // const unsigned long long int   lambda=9;
  // const unsigned long long int       mu=3;

  // generatore "ragionevole" ma che presenta l'effetto Marsaglia
  // const unsigned long long int m=pow(2,12);
  // const unsigned long long int   lambda=17;
  // const unsigned long long int        mu=3;

  // generatore "ragionevole" ma che presenta l'effetto Marsaglia
  // const unsigned long long int m=pow(2,12);
  // const unsigned long long int    lambda=9;
  // const unsigned long long int        mu=5;

  // generatore "ragionevole" ma che presenta l'effetto Marsaglia
  const unsigned long long int m=pow(2,12);
  const unsigned long long int    lambda=5;
  const unsigned long long int        mu=3;
  
  //**********************************************************
  
  int algo;
  
  TRandom* trand;  
  TRandom* trand1;
  TRandom* trand2;
  TRandom* trand3;
  
public:
  RndGen(unsigned long long int seed=seeddefault);

  inline void SetAlgorithm(int algorithm) { algo=algorithm; };
  static std::string GetAlgoName(int algo);
  
  void SetSeed(unsigned long long int seed);
  inline unsigned long long int GetRandom() { return random; };
  unsigned long long int GetRandomAndThrowANewOne();
  unsigned long long int GetMaximumNumber();

  double GetRandomUniformAndThrowANewOne(double xlow=0.0, double xhigh=1.0);
  double GetRandomExponentialAndThrowANewOne(double par0);
  double GetRandomGaussianAndThrowANewOne(double par0, double par1);

private:
  unsigned long long int GetRandomAndThrowANewOneSimple();
  unsigned long long int GetRandomAndThrowANewOneLCG();
  unsigned long long int GetRandomAndThrowANewOneTRandom();
  unsigned long long int GetRandomAndThrowANewOneTRandom1();
  unsigned long long int GetRandomAndThrowANewOneTRandom2();
  unsigned long long int GetRandomAndThrowANewOneTRandom3();

  static unsigned long long int GetMaximumNumberSimple();
  inline unsigned long long int GetMaximumNumberLCG() { return GetMaximumNumberLCG(m); }; 
  static unsigned long long int GetMaximumNumberLCG(unsigned long long int _modulus);
  static unsigned long long int GetMaximumNumberTRandom();
  static unsigned long long int GetMaximumNumberTRandom1();
  static unsigned long long int GetMaximumNumberTRandom2();
  static unsigned long long int GetMaximumNumberTRandom3();
};

RndGen::RndGen(unsigned long long int seed) {
  
  algo=1;
  
  trand  = new TRandom();
  trand1 = new TRandom1();
  trand2 = new TRandom2();
  trand3 = new TRandom3();
  
  SetSeed(seed);
  
  return;
}

void RndGen::SetSeed(unsigned long long int seed) {

  switch (algo) {
  case 0:
    random=seed;
    break;
  case 1:
    random=seed;
    break;
  case 2:
    trand->SetSeed(seed);
    GetRandomAndThrowANewOneTRandom();
    break;
  case 3:
    trand1->SetSeed(seed);
    GetRandomAndThrowANewOneTRandom1();
    break;
  case 4:
    trand2->SetSeed(seed);
    GetRandomAndThrowANewOneTRandom2();
    break;
  case 5:
    trand3->SetSeed(seed);
    GetRandomAndThrowANewOneTRandom3();
    break;
  }
  
  return;
};

unsigned long long int RndGen::GetRandomAndThrowANewOne(){

  switch (algo) {
  case 0:
    return GetRandomAndThrowANewOneSimple();
  case 1:
    return GetRandomAndThrowANewOneLCG();
  case 2:
    return GetRandomAndThrowANewOneTRandom();
  case 3:
    return GetRandomAndThrowANewOneTRandom1();
  case 4:
    return GetRandomAndThrowANewOneTRandom2();
  case 5:
    return GetRandomAndThrowANewOneTRandom3();
  }

  return random;
}

unsigned long long int RndGen::GetMaximumNumber(){

  // questo ha senso solo per il SimpleGenerator e per l'LCG
  switch (algo) {
  case 0:
    return GetMaximumNumberSimple();
  case 1:
    return GetMaximumNumberLCG();
  case 2:
    return GetMaximumNumberTRandom();
  case 3:
    return GetMaximumNumberTRandom1();
  case 4:
    return GetMaximumNumberTRandom2();
  case 5:
    return GetMaximumNumberTRandom3();
  }
  
  return 0;
}

inline unsigned long long int RndGen::GetMaximumNumberSimple() {
  return (unsigned long long int)(pow(2,16)-1);
};

inline unsigned long long int RndGen::GetMaximumNumberLCG(unsigned long long int _modulus) {
  return _modulus-1;
};

inline unsigned long long int RndGen::GetMaximumNumberTRandom() {
  //  return (unsigned long long int)(pow(2,8*sizeof(unsigned long long int))-1);
  return (unsigned long long int)(-1);//the pow is not "enough" so I cast "-1" to obtain the maximum number I can write with the unsigned; 
};

inline unsigned long long int RndGen::GetMaximumNumberTRandom1() {
  //  return (unsigned long long int)(pow(2,8*sizeof(unsigned long long int))-1);
  return (unsigned long long int)(-1);//the pow is not "enough" so I cast "-1" to obtain the maximum number I can write with the unsigned; 
};

inline unsigned long long int RndGen::GetMaximumNumberTRandom2() {
  //  return (unsigned long long int)(pow(2,8*sizeof(unsigned long long int))-1);
  return (unsigned long long int)(-1);//the pow is not "enough" so I cast "-1" to obtain the maximum number I can write with the unsigned; 
};

inline unsigned long long int RndGen::GetMaximumNumberTRandom3() {
  //  return (unsigned long long int)(pow(2,8*sizeof(unsigned long long int))-1);
  return (unsigned long long int)(-1);//the pow is not "enough" so I cast "-1" to obtain the maximum number I can write with the unsigned; 
};

unsigned long long int RndGen::GetRandomAndThrowANewOneSimple(void) {
  
  // Aggiorna sequenza random
  // Algoritmo Polinomiale:
  // +> b0 b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15
  // |     |  |     |                                     |
  // ------+--+-----+-------------------------------------+
  // carry = b1^b2^b4^b15
  // Pn+1=(Pn<<1)|carry
  
  unsigned short int randomtmp; // Accumulo le operazioni ex-OR
  if (random==0) random++;      // N.B. : il seed dovrebbe essere != 0
  randomtmp=0;
  if ((unsigned short int)random&0x02  ) randomtmp =1;
  if ((unsigned short int)random&0x04  ) randomtmp^=1;
  if ((unsigned short int)random&0x10  ) randomtmp^=1;
  if ((unsigned short int)random&0x8000) randomtmp^=1;
  random = (unsigned short int)((random<<1)|randomtmp);

  return random;
}

unsigned long long int RndGen::GetRandomAndThrowANewOneLCG(void) {

  random = (lambda*random + mu)%m;

  return random;
}

inline unsigned long long int RndGen::GetRandomAndThrowANewOneTRandom() {
  random = (unsigned long long int)(pow(2,8*sizeof(unsigned long long int))*(trand->Uniform()));
  return random;
};

inline unsigned long long int RndGen::GetRandomAndThrowANewOneTRandom1()  {
  random = (unsigned long long int)(pow(2,8*sizeof(unsigned long long int))*(trand1->Uniform()));
  return random;
};

inline unsigned long long int RndGen::GetRandomAndThrowANewOneTRandom2()  {
  random = (unsigned long long int)(pow(2,8*sizeof(unsigned long long int))*(trand2->Uniform()));
  return random;
};

inline unsigned long long int RndGen::GetRandomAndThrowANewOneTRandom3()  {
  random = (unsigned long long int)(pow(2,8*sizeof(unsigned long long int))*(trand3->Uniform()));
  return random;
};

std::string RndGen::GetAlgoName(int algo) {

  switch(algo) {
  case 0:
    return "Simple";
  case 1:
    return "LCG";
  case 2:
    return "TRandom";
  case 3:
    return "TRandom1";
  case 4:
    return "TRandom2";
  case 5:
    return "TRandom3";
  }

  return "";
}

double RndGen::GetRandomUniformAndThrowANewOne(double xlow, double xhigh){
  unsigned long long int _random = GetRandomAndThrowANewOne();
  unsigned long long int _modulus = GetMaximumNumber();
  double _uniform = xlow+((xhigh-xlow)*_random)/((double)_modulus+1);//the +1 is needed to make [xlow, xhigh[
  //  printf("%llu %llu %f %f --> %f\n", _random, _modulus, xlow, xhigh, _uniform);
  return _uniform;
}

double RndGen::GetRandomExponentialAndThrowANewOne(double par0){
  double _uniform = GetRandomUniformAndThrowANewOne();
  double _exponential = -par0*log(1.0-_uniform);
  //  printf("%f %f --> %f --> %f\n", _uniform, par0, log(1.0-_uniform), _exponential);
  return _exponential;
}

double RndGen::GetRandomGaussianAndThrowANewOne(double par0, double par1){

  static double _gaussian;
  static bool _unloaded=true;
  
  if (_unloaded) {
    double _uniform = GetRandomUniformAndThrowANewOne();
    double _rho = sqrt(-2.0*log(1.0-_uniform));
    double _phi = GetRandomUniformAndThrowANewOne(0.0, 2.0*TMath::Pi());
    
    double _gaussian1 = _rho*cos(_phi);
    double _gaussian2 = _rho*sin(_phi);

    _gaussian=_gaussian2;
    _unloaded=false;
    return par0+par1*_gaussian1;
  }
  else {
    _unloaded=true;
    return par0+par1*_gaussian;
  }
}

void testarbitrary(RndGen* rndgen, int algo, int ifunc, double par0, double par1, unsigned long long int entries, unsigned long long int seed){

  std::string s_algo = RndGen::GetAlgoName(algo);

  rndgen->SetAlgorithm(algo);
  rndgen->SetSeed(seed);
  
  TH1F* hist = new TH1F("hist", "histogram; X (u.a.); Entries", 100, 0, 100);

  for (unsigned long long int ii=0; ii<entries; ii++) {
    double _random=0.0;
    if (ifunc==0)      _random=rndgen->GetRandomUniformAndThrowANewOne(par0, par1);
    else if (ifunc==1) _random=rndgen->GetRandomExponentialAndThrowANewOne(par0);
    else if (ifunc==2) _random=rndgen->GetRandomGaussianAndThrowANewOne(par0, par1);
    //    printf("%f\n", _random);
    hist->Fill(_random);
  }

  std::string s_func;
  switch (ifunc) {
  case 0:
    s_func="Uniform";
    break;
  case 1:
    s_func="Exponential";
    break;
  case 2:
    s_func="Gaussian";
    break;
  }
  
  TCanvas* c = new TCanvas((s_func+s_algo).c_str(), (s_func+s_algo).c_str(), 700, 600);
  c->cd();
  hist->DrawCopy();
  if (hist) delete hist;
  
  c->Modified();
  c->Update();
  
  return;  
}

void testperiod(RndGen* rndgen, int algo, unsigned long long int seed) {

  printf("%s\n", RndGen::GetAlgoName(algo).c_str());
  
  unsigned long long int old;
  unsigned long long int first;
  unsigned long long int second;

  //  printf("seed = %llu, random=%llu\n", seed, rndgen->GetRandom());
  rndgen->SetAlgorithm(algo);
  //  printf("seed = %llu, random=%llu\n", seed, rndgen->GetRandom());
  rndgen->SetSeed(seed);
  //  printf("seed = %llu, random=%llu\n", seed, rndgen->GetRandom());
  
  first = rndgen->GetRandom();
  old = first;
  second= rndgen->GetRandomAndThrowANewOne();
  //  printf("seed=%lld, first=%llu, second=%llu, old=%llu\n", seed, first, second, old);
  
  for (unsigned long long int ii=0; ii<=std::min(rndgen->GetMaximumNumber(), ((unsigned long long int)pow(2,32)))+1; ii++) {
    unsigned long long int _rnd = rndgen->GetRandom();
    if (ii!=0 && _rnd==second) {
      //printo anche il numero precedente: TRandom1,2 e 3 non sono LCG! Si ripete un numero ma non la sequenza!
      printf("%llu) random=%llu, old=%llu (second=%llu, first=%llu)\n", ii, _rnd, old, second, first);
    }
    old=_rnd;
    rndgen->GetRandomAndThrowANewOne();
  }

  printf("*******************\n");
  
  return;
}

void testmarsaglia(RndGen* rndgen, int algo, unsigned long long int seed) {

  system("mkdir -p ./Immagini");
  
  std::string s_algo = RndGen::GetAlgoName(algo);
  
  printf("%s\n", s_algo.c_str());

  rndgen->SetAlgorithm(algo);
  rndgen->SetSeed(seed);
  
  TGraph* gr = new TGraph();
  gr->SetMarkerStyle(20);
  gr->SetMarkerSize(1.0);

  TCanvas* c = new TCanvas(s_algo.c_str(), s_algo.c_str(), 700, 600);
  c->cd();
  TH1F* frame = c->DrawFrame(0.0, 0.0, rndgen->GetMaximumNumber(), rndgen->GetMaximumNumber());
  frame->GetXaxis()->SetTitle("x (u.a.)");
  frame->GetYaxis()->SetTitle("Y (u.a.)");
  gr->Draw("P");

  std::string s_algo_gif = "Immagini/"+s_algo+".gif";
  char animfile[100];
  sprintf(animfile,"%s+1",s_algo_gif.c_str());
  
  for (unsigned long long int ii=1; ii<=(std::min(rndgen->GetMaximumNumber(), (unsigned long long int)pow(2,16))+1); ii++) {
    unsigned long long int x = rndgen->GetRandomAndThrowANewOne();
    unsigned long long int y = rndgen->GetRandomAndThrowANewOne();
    gr->SetPoint(ii, x, y);
    if (
	std::floor(log2(ii))==log2(ii) ||
	std::floor(log2(ii+1))==log2(ii+1) ||
	std::floor(log2(ii+2))==log2(ii+2)
	) {
      //      printf("%llu) %llu %llu\n", ii, x, y);
      c->Modified();
      c->Update();
      c->Print(animfile);
    }
  }
  gr->Print();

  sprintf(animfile,"%s++",s_algo_gif.c_str()); // infinite loop
  c->Print(animfile);
  
  std::string s_algo_pdf = "Immagini/"+s_algo+".pdf";
  c->SaveAs(s_algo_pdf.c_str(), "pdf");

  //  if (gr) delete gr;
  //  if (c) delete c;
  
  printf("*******************\n");
  
  return;
}

int main(int argc, char* argv[]) {

  TApplication *myapp=new TApplication("myapp",0,0);
  
  RndGen* rndgen = new RndGen();

  // qui si testa la generazione di numeri random con p.d.f arbitrarie
  // - con LCG (algo=1)
  // * per la Uniforme (ifunc=0) in [0, 85[
  // * per l'esponenziale (ifunc=1) e^{-5}
  // * per la Gaussiana (ifunc=2) μ=50, σ=10
  testarbitrary(rndgen, 2, 0, 10, 85, 1000000);
  testarbitrary(rndgen, 2, 1,  5,  0, 1000000);
  testarbitrary(rndgen, 2, 2, 50, 10, 1000000);

  // qui si testa la periodicità del generatore
  // * per il generatore semplice a 16 bit (algo=0)
  // * per LCG (algo=1)
  // * per TRandom (algo=2)
  // * per TRandom3 (algo=3): non si riesce a trovare e non basta cercare la ripetizione del singolo numero!
  testperiod(rndgen, 0);
  testperiod(rndgen, 1);
  testperiod(rndgen, 2);
  testperiod(rndgen, 5);

  // qui si ricerca l'effetto Marsaglia in 2D
  // * per il generatore semplice a 16 bit (algo=0)
  // * per LCG (algo=1)
  // * per TRandom (algo=2)
  testmarsaglia(rndgen, 0);
  testmarsaglia(rndgen, 1);
  testmarsaglia(rndgen, 2);
  
  myapp->Run();
  
  return 0;
}
