#include <iostream>
#include <cmath>
#include <chrono>
#include <vector>
#include <algorithm>
//
// Questo programma calcola l'integrale di una funzione
// con i metodi del trapezio, di Simpson e di Gauss
//

#define TRIALSFORPROFILING 1000
#define REPETITIONFORPROFILING 100

typedef double (*funcp)(double, double, double f(double), unsigned int);

// funzione integranda
double funpol2(double);
double funpol3(double);
double funpol4(double);
double funpol5(double);
double funosc(double);
double funexp(double);
double fungauss(double);
double funrunge(double);

//utilities
void testametodo(const std::string method, const std::string function, double f(double), double xmin, double xmax, unsigned int npoints, double exactint);
double timeprofilametodo(funcp fi, double f(double), double xmin, double xmax, unsigned int npoints);

//metodi di integrazione
double integraleTrapezio(double xmin, double xmax, double f(double), unsigned int npoints);
double integraleSimpson( double xmin, double xmax, double f(double), unsigned int npoints);
double integraleGauss(   double xmin, double xmax, double f(double), unsigned int npoints);

double integraleTrapezio(double xmin, double xmax, double f(double), unsigned int n) {

  // intervallo tra due punti
  double h=(xmax-xmin)/n;
  
  // l'integrale e' una somma, all'inizio va azzerato
  double sum=0.;
  for(unsigned int j=0; j<=n; j++) {
    // calcolo l'ascissa del punto
    double x=xmin+j*h;
    // valuto la funzione e la aggiungo all'integrale
    if(j==0 || j==n) sum += 0.5*f(x);
    else sum += f(x);
  }
  sum *= h;
  
  return sum;
}

double integraleSimpson(double xmin, double xmax, double f(double), unsigned int n) {
  
  unsigned int duen = (unsigned int)(2*n);
  
  // intervallo tra due punti in cui valuto la funzione
  double h=(xmax-xmin)/double(duen);
  
  // sommo i termini pari
  double sum1=0.;
  for(unsigned int j=0; j<=duen; j += 2) {
    double x=xmin+j*h;
    sum1 += f(x);
  }
  
  // sommo i termini dispari
  double sum2=0.;
  for(unsigned int j=1; j<=duen; j += 2) {
    double x=xmin+j*h;
    sum2=sum2+f(x);
  }
  
  // sottraggo 1/3 degli estremi e moltiplico per h
  double sum=h*(2./3.*sum1+4./3.*sum2-1./3.*(f(xmin)+f(xmax)));
  
  return sum;
}

double integraleGauss(double xmin, double xmax, double f(double), unsigned int n) {

  double x;
  double somma=0.0;
  double* y = new double[n+1];
  double* w = new double[n+1];
  
  // ascisse e pesi sono tabulati, non e' necessario calcolarli
  // le ascisse sono antisimmetriche i pesi simmetrici
  if (n==10) {
    // ascisse
    y[0]  =  0;
    y[1]  =  0.1488743389;
    y[2]  =  0.4333953941;
    y[3]  =  0.6794095682;
    y[4]  =  0.8650633666;
    y[5]  =  0.9739065285;
    y[6]  = -0.9739065285;
    y[7]  = -0.8650633666;
    y[8]  = -0.6794095682;
    y[9]  = -0.4333953941;
    y[10] = -0.1488743389;
    // pesi
    w[0]  =  0;
    w[1]  =  0.2955242247;
    w[2]  =  0.2692667193;
    w[3]  =  0.2190863625;
    w[4]  =  0.1494513491;
    w[5]  =  0.0666713443;
    w[6]  =  0.0666713443;
    w[7]  =  0.1494513491;
    w[8]  =  0.2190863625;
    w[9]  =  0.2692667193;
    w[10] =  0.2955242247;
  }
  else if (n==20) {
    // ascisse
    y[0]  =  0;
    y[1]  =  7.6526521133497e-2;
    y[2]  =  2.27785851141645e-1;
    y[3]  =  3.73706088715420e-1;
    y[4]  =  5.10867001950827e-1;
    y[5]  =  6.36053680726515e-1;
    y[6]  =  7.46331906460151e-1;
    y[7]  =  8.39116971822219e-1;
    y[8]  =  9.12234428251326e-1;
    y[9]  =  9.63971927277914e-1;
    y[10] =  9.93128599185095e-1;
    y[11] = -9.93128599185095e-1;
    y[12] = -9.63971927277914e-1;
    y[13] = -9.12234428251326e-1;
    y[14] = -8.39116971822219e-1;
    y[15] = -7.46331906460151e-1;
    y[16] = -6.36053680726515e-1;
    y[17] = -5.10867001950827e-1;
    y[18] = -3.73706088715420e-1;
    y[19] = -2.27785851141645e-1;
    y[20] = -7.6526521133497e-2;
    // pesi
    w[0]  = 0;
    w[1]  = 1.52753387130726e-1;
    w[2]  = 1.49172986472604e-1;
    w[3]  = 1.42096109318382e-1;
    w[4]  = 1.31688638449177e-1;
    w[5]  = 1.18194531961518e-1;
    w[6]  = 1.01930119817240e-1;
    w[7]  = 8.3276741576705e-2;
    w[8]  = 6.2672048334109e-2;
    w[9]  = 4.0601429800387e-2;
    w[10] = 1.7614007139152e-2;
    w[11] = 1.7614007139152e-2;
    w[12] = 4.0601429800387e-2;
    w[13] = 6.2672048334109e-2;    
    w[14] = 8.3276741576705e-2;
    w[15] = 1.01930119817240e-1;
    w[16] = 1.18194531961518e-1;
    w[17] = 1.31688638449177e-1;
    w[18] = 1.42096109318382e-1;
    w[19] = 1.49172986472604e-1;
    w[20] = 1.52753387130726e-1;
  }

  if (n==10 || n==20) {
    // cambio di variabili per integrare tra a e b
    // invece che tra 0 e 1
    for(int j=1; j<=n; j++) {
      x=0.5*(xmax+xmin)+0.5*(xmax-xmin)*y[j];
      somma=somma+w[j]*f(x);
    }
    somma = (xmax-xmin)/2.*somma;
  }
  else {
    printf("Metodo di Gauss non implementato per n=%d\n", n);
  }

  delete[] y;
  delete[] w;
  
  return somma;
}

double funpol2(double x) {
  // funzione polinomiale
  return pow(x,2);
}

double funpol3(double x) {
  // funzione polinomiale
  return pow(x,3);
}

double funpol4(double x) {
  // funzione polinomiale
  return pow(x,3)-pow(x,4);
}

double funpol5(double x) {
  // funzione polinomiale
  return 4*pow(x,3)-5*pow(x,4)+6*pow(x,5);
}

double funosc(double x) {
  // funzione oscillante
  return sin(10*x);
}

double funexp(double x) {
  // funzione esponenziale
  return exp(-5*x);
}

double fungauss(double x) {
  // funzione gaussiana
  static const double PI = 4.0 * atan(1.0);
  return exp(-x*x/2)/sqrt(2*PI);
}

double funrunge(double x) {
  //funzione di Runge
  return 1.0/(1.0+25*x*x);
}

double timeprofilametodo(funcp fi, double f(double), double xmin, double xmax, unsigned int npoints){

  //  using namespace std::chrono;
  
  std::chrono::high_resolution_clock::time_point t1;
  std::chrono::high_resolution_clock::time_point t2;

  std::vector<double> vec;
  vec.reserve(TRIALSFORPROFILING);
  
  for (int ii=0; ii<TRIALSFORPROFILING; ii++) {
    t1 = std::chrono::high_resolution_clock::now();
    for (int jj=0; jj<REPETITIONFORPROFILING; jj++) {
      fi(xmin, xmax, f, npoints);
    }
    t2 = std::chrono::high_resolution_clock::now();
    auto duration = std::chrono::duration_cast<std::chrono::microseconds>(t2 - t1).count();
    //    printf("%d %d %f\n", (unsigned int)duration, REPETITIONFORPROFILING, (double)(duration)/REPETITIONFORPROFILING);
    vec.push_back((double)(duration)/REPETITIONFORPROFILING);
  }

  const auto median_it1 = vec.begin() + vec.size() / 2 - 1;
  const auto median_it2 = vec.begin() + vec.size() / 2;
  std::nth_element(vec.begin(), median_it1 , vec.end()); // e1
  std::nth_element(vec.begin(), median_it2 , vec.end()); // e2
  double median = (vec.size()%2==0)?(*median_it1+*median_it2)/2:*median_it2;
  
  return median;
}

void testametodo(const std::string method, const std::string function, double f(double), double xmin, double xmax, unsigned int npoints, double exactint) {
    
  funcp fi;
  
  if (method=="Trapezio")     fi = integraleTrapezio;
  else if (method=="Simpson") fi = integraleSimpson;
  else if (method=="Gauss")   fi = integraleGauss;
  else {
    printf("metodo %s non implementato\n", method.c_str());
    return;
  }

  double numintLO = fi(xmin, xmax, f, npoints);
  double durationLO  = timeprofilametodo(fi, f, xmin, xmax, npoints);
  
  double numintNLO = fi(xmin, xmax, f, (unsigned int)(2*npoints));
  double durationNLO = timeprofilametodo(fi, f, xmin, xmax, (unsigned int)(2*npoints));
  
  printf("%10s - %25s:  \t", method.c_str(), function.c_str());
  printf("%+f --> %E (%E) \t (%.3f ms - %.3f ms) \n", numintLO, fabs(numintLO-exactint), fabs(numintLO-numintNLO), durationLO, durationNLO);

  return;  
}

int main(int argc, char* argv[]) {

  double exactintfunpol2 = 8.0/3.0;
  testametodo("Trapezio", "funzione polinomiale (2)", funpol2, 0.0, 2.0, 100, exactintfunpol2);
  testametodo("Simpson",  "funzione polinomiale (2)", funpol2, 0.0, 2.0, 100, exactintfunpol2);
  testametodo("Gauss",    "funzione polinomiale (2)", funpol2, 0.0, 2.0,  10, exactintfunpol2);

  printf("***************************************************\n");
  
  double exactintfunpol3 = 1.0/64.0;
  testametodo("Trapezio", "funzione polinomiale (3)", funpol3, 0.0, 0.5, 100, exactintfunpol3);
  testametodo("Simpson",  "funzione polinomiale (3)", funpol3, 0.0, 0.5, 100, exactintfunpol3);
  testametodo("Gauss",    "funzione polinomiale (3)", funpol3, 0.0, 0.5,  10, exactintfunpol3);

  printf("***************************************************\n");
  
  double exactintfunpol4 = 1.0/20.0;
  testametodo("Trapezio", "funzione polinomiale (4)", funpol4, 0.0, 1.0, 100, exactintfunpol4);
  testametodo("Simpson",  "funzione polinomiale (4)", funpol4, 0.0, 1.0, 100, exactintfunpol4);
  testametodo("Gauss",    "funzione polinomiale (4)", funpol4, 0.0, 1.0,  10, exactintfunpol4);

  printf("***************************************************\n");

  double exactintfunpol5 = 1.0;
  testametodo("Trapezio", "funzione polinomiale (5)", funpol5, 0.0, 1.0, 100, exactintfunpol5);
  testametodo("Simpson",  "funzione polinomiale (5)", funpol5, 0.0, 1.0, 100, exactintfunpol5);
  testametodo("Gauss",    "funzione polinomiale (5)", funpol5, 0.0, 1.0,  10, exactintfunpol5);

  printf("***************************************************\n");

  const double PI = 4.0 * atan(1.0);
  double exactintfunosc = 0.0;
  testametodo("Trapezio", "funzione sinusoidale", funosc, 0.0, PI, 100, exactintfunosc);
  testametodo("Simpson",  "funzione sinusoidale", funosc, 0.0, PI, 100, exactintfunosc);
  testametodo("Gauss",    "funzione sinusoidale", funosc, 0.0, PI,  10, exactintfunosc);

  printf("***************************************************\n");

  const double NEP = exp(1);
  double exactintfunexp = (NEP-1)/(5*NEP);
  testametodo("Trapezio", "funzione esponenziale", funexp, 0.0, 1.0/5.0, 100, exactintfunexp);
  testametodo("Simpson",  "funzione esponenziale", funexp, 0.0, 1.0/5.0, 100, exactintfunexp);
  testametodo("Gauss",    "funzione esponenziale", funexp, 0.0, 1.0/5.0,  10, exactintfunexp);

  printf("***************************************************\n"); 
  
  double exactintfungauss = 0.682689492137;
  testametodo("Trapezio", "funzione Gaussiana", fungauss, -1.0, 1.0, 100, exactintfungauss);
  testametodo("Simpson",  "funzione Gaussiana", fungauss, -1.0, 1.0, 100, exactintfungauss);
  testametodo("Gauss",    "funzione Gaussiana", fungauss, -1.0, 1.0,  10, exactintfungauss);

  printf("***************************************************\n"); 
  
  double exactintfunrunge = 1.0/5.0*(atan(5)-atan(-5));
  testametodo("Trapezio", "funzione di Runge", funrunge, -1.0, 1.0, 100, exactintfunrunge);
  testametodo("Simpson",  "funzione di Runge", funrunge, -1.0, 1.0, 100, exactintfunrunge);
  testametodo("Gauss",    "funzione di Runge", funrunge, -1.0, 1.0,  10, exactintfunrunge);
  
  
  return 0;
}
